<?php
/**
 * Create an ACF form, and use hooks to handle 
 * the creation of a new custom post.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

//================================================================================
// FORM SHORTCODE DEFINITION
//================================================================================

/**
 * This function contains all the code needed 
 * to render the ACF form for submitting membership requests, 
 * in the form of a WordPress shortcode.
 */
function eruahaus_membership_requests_form_shortcode( $atts ) {

    extract( shortcode_atts( array(), $atts ) );
    $output = '';

    $user = wp_get_current_user();
    $disallowed_roles = array('eruaner');

    ob_start();

    // Show the form only to logged-in users
    if ( is_user_logged_in() && $user && ! array_intersect($disallowed_roles, $user->roles) ) {

        $user = wp_get_current_user();

        $user_requests = get_posts( [
            'post_type'   => 'membership-request',
            'fields' => 'ids',
            'post_status' => ['publish', 'pending', 'draft', 'future', 'private', 'inherit'],
            'author'      => $user->ID,
            'orderby'     => 'date',
            'numberposts' => 1,
            // 'tax_query' => [
            //     [
            //         'taxonomy' => 'membership-request-status',
            //         'field' => 'slug',
            //         'terms' => ['closed', 'approved'],
            //         'operator' => 'NOT IN',
            //     ],
            // ],
        ]);
        if (!empty($user_requests)) wp_reset_postdata();
        //$latest_request = $user_requests[0];
        $count = count($user_requests);

        if ( $count >=1 ) { # && !current_user_can('administrator')
            echo '<div class="erua-notice info">'. __('You have already a pending request for membership. Please contact the website\'s team for any inquiries regarding your membership request.', 'erua') .'</div>';
        }
        else {
            echo '<div id="publish"><h2 id="submit-request" class="acf-form-heading">'.__('Apply to become an ERUAner!', 'erua').'</h2>';
                    
            // https://www.advancedcustomfields.com/resources/acf_register_form/
            $settings = array(

                /* (string) Unique identifier for the form. Defaults to 'acf-form' */
                'id' => 'eruahaus-membership-requests-form',

                /* (int|string) The post ID to load data from and save data to. Defaults to the current post ID.
                Can also be set to 'new_post' to create a new post on submit */
                //'post_id'		=> "user_" . $current_user->ID,
                'post_id'		=> 'new_post',

                /* (array) An array of post data used to create a post. See wp_insert_post for available parameters.
                The above 'post_id' setting must contain a value of 'new_post' */
                // 'new_post'	=> array(
                //     'post_type'		=> 'request',
                //     'post_status'	=> 'publish'
                // ),

                /* (array) An array of field group IDs/keys to override the fields displayed in this form */
                //'field_groups' => array('group_5a16ce3c67cb6', 'group_59f6eb9e7c014'),
                'field_groups' => array('group_6555781f1e082'),

                /* (array) An array of field IDs/keys to override the fields displayed in this form */
                //'fields' => false,

                /* (boolean) Whether or not to show the post title text field. Defaults to false */
                'post_title' => false,

                /* (boolean) Whether or not to show the post content editor field. Defaults to false */
                'post_content' => false,

                /* (boolean) Whether or not to create a form element. Useful when a adding to an existing form. Defaults to true */
                'form' => true,
                                                    
                /* (string) The URL to be redirected to after the form is submit. Defaults to the current URL with a GET parameter '?updated=true'.
                A special placeholder '%post_url%' will be converted to post's permalink (handy if creating a new post)
                A special placeholder '%post_id%' will be converted to post's ID (handy if creating a new post) */
                //'return' => '?updated=true#library-form-heading',
                'return'	=> site_url('membership-request-submitted'), // Redirect to a thank you page
                //'return' => add_query_arg( 'updated', 'true', get_permalink() )

                /* (string) Extra HTML to add before the fields */
                //'html_before_fields' => '',

                /* (string) Extra HTML to add after the fields */
                'html_after_fields'  => '',
                //'html_after_fields'  => '<input type="hidden" id="project-categories-input" name="project-categories" value="' . implode(",", $project_categories) . '" />',

                /* (string) The text displayed on the submit button */
                'submit_value' => __('Submit', 'erua'),

                /* (string) A message displayed above the form after being redirected. Can also be set to false for no message */
                'updated_message' => __('Your Membership Request has been submitted!', 'erua'),

                /* (string) Determines where field labels are places in relation to fields. Defaults to 'top'.
                Choices of 'top' (Above fields) or 'left' (Beside fields) */
                'label_placement' => 'top',

                /* (string) Determines where field instructions are places in relation to fields. Defaults to 'label'.
                Choices of 'label' (Below labels) or 'field' (Below fields) */
                'instruction_placement' => 'label',

                /* (string) Determines element used to wrap a field. Defaults to 'div'
                Choices of 'div', 'tr', 'td', 'ul', 'ol', 'dl' */
                'field_el' => 'div',

                /* (string) Whether to use the WP uploader or a basic input for image and file fields. Defaults to 'wp'
                Choices of 'wp' or 'basic'. Added in v5.2.4 */
                'uploader' => 'wp'

                /* (boolean) Whether to include a hidden input field to capture non human form submission. Defaults to true. Added in v5.3.4 */
                //'honeypot' => true,

                /* (string) HTML used to render the updated message. Added in v5.5.10 */
                //'html_updated_message'	=> '<div id="message" class="updated"><p>%s</p></div>',

                /* (string) HTML used to render the submit button. Added in v5.5.10 */
                //'html_submit_button'	=> '<input type="submit" class="acf-button button button-primary button-large" value="%s" />',

                /* (string) HTML used to render the submit button loading spinner. Added in v5.5.10 */
                //'html_submit_spinner'	=> '<span class="acf-spinner"></span>',

                /* (boolean) Whether or not to sanitize all $_POST data with the wp_kses_post() function. Defaults to true. Added in v5.6.5 */
                //'kses'	=> true
            );

            acf_form($settings);

            echo '</div>';
        }

    }

    $output = ob_get_contents();
    ob_end_clean();

    return $output;
}
add_shortcode('eruahaus-membership-requests-form', 'eruahaus_membership_requests_form_shortcode');

//================================================================================
// FORM PROCESSING
//================================================================================

/**
 * Using a hook to act before an ACF form is saved,
 * we will cancel saving if conditions for a new post are not met, 
 * and if they are, we'll then get the POST data 
 * and insert it into the new post.
 */
function eruahaus_membership_requests_acf_form_pre_save_post( $post_id ) {

    // Stop function and bail early if not a new post
    if( $post_id !== 'new_post' ) {
        return $post_id;
    }

    if ( isset($_POST['acf']['field_655aa1c002e35']) && $_POST['acf']['field_655aa1c002e35'] === 'eruahaus-membership-request' ) {

        // echo '<pre>';
        // var_dump($_POST);
        // echo '</pre>';
        // die();

        $current_user = wp_get_current_user();

        // If the user couldn't be fetched then abort
        if ( ! $current_user ) return $post_id;

        $title = $current_user->display_name . ' (' . $current_user->user_email . ')';

        // Get "Open" status (custom taxonomies/categories)
        $status_term = get_term_by('id', 150, 'membership-request-status');
        $status_slug = $status_term->slug;
        $status = (int) $status_term->term_id;

        /**
         * Create a new post
         * https://developer.wordpress.org/reference/functions/wp_insert_post/#parameters
         * https://wordpress.stackexchange.com/questions/78543/wp-insert-post-or-wp-set-post-terms-do-not-save-taxonomy-but-wp-set-post-terms
         * https://wordpress.org/support/topic/polylangcount-posts-using-get_term_by/
         */
        $post = array(
            'post_status'	=> 'publish',
            'post_type'		=> 'membership-request',
            'post_title'	=> $title,
            'post_author'	=> $current_user->ID,
            //'comment_status' => 'closed',
            //'post_content'  => $description,
            // for core categories taxonomy only
            //'post_category' => array($category),
            // for custom taxonomies
            'tax_input'     => array(
                'membership-request-status' => array( intval( $status ) )
            ),
        );

        // Insert the post
        $post_id = wp_insert_post( $post );

        // Make it Private (private posts don't appear in the taxonomy term count!)
        wp_update_post( array( 'ID' => $post_id, 'post_status' => 'private' ));

        // Set its default status to "Open"
        $term_taxonomy_ids = wp_set_object_terms( $post_id, array( intval( $status ) ), 'membership-request-status' );

        /**
         * Insert Applicant's Name
         */
        update_field( 'field_6554ffe7792ec', $current_user->display_name, $post_id );

        /**
         * Insert Applicant's e-mail
         */
        update_field( 'field_6554ffea792ed', $current_user->user_email, $post_id );

        /**
         * Insert Applicant's User ID
         */
        update_field( 'field_6554fff8792ee', $current_user->ID, $post_id );

        /**
         * Insert Applicant's Profile URL
         */
        if ( function_exists('bp_is_active') ) {
            update_field( 'field_655508ced6e48', bbp_get_user_profile_url( (int) $current_user->ID ), $post_id );
        }

        // Send e-mail notifications
        //send_emails_after_membership_request_creation($post_id);
    }

    // Return the new ID
    return $post_id;

}
add_filter('acf/pre_save_post' , 'eruahaus_membership_requests_acf_form_pre_save_post', 1, 1);

/**
 * Hooking into 'acf/save_post' with a high priority such as 20,
 * ensures that the $_POST data have already been saved,
 * and therefore we can get/access all custom fields of the post.
 * Alternatively we may use a priority lower than 10,
 * to manipulate the data before it gets saved.
 * 
 * https://www.advancedcustomfields.com/resources/acf-save_post/
 */
function eruahaus_membership_requests_acf_form_after_save_post( $post_id ) {

    if ( get_post_type($post_id) !== 'membership-request' ) {
        return;
    }

    $user_id = get_field('eruahaus_membership_request_registered_user_id', $post_id);

    if ( !$user_id ) {
        return;
    }

    $user = get_user_by('id', (int) $user_id);

    $terms = get_the_terms($post_id, 'membership-request-status');
    $current_status = $terms[0]->slug;

    if ( $current_status === 'approved' && !in_array( 'eruaner', (array) $user->roles ) ) {
        $user->add_role( 'eruaner' );
        $user_bp_member_types = bp_get_member_type( $user_id, false );
        if ( !in_array( 'eruaner', (array) $user_bp_member_types ) ) {
            bp_set_member_type( $user_id, 'eruaner', true );
        }
        //send_emails_after_membership_granted($post_id, $user_id);
    }
}
add_action('acf/save_post', 'eruahaus_membership_requests_acf_form_after_save_post', 20);

//================================================================================
// FIELD HANDLING - FORM IDENTIFIER
//================================================================================

/**
 * Detect any change in the default value of the form identifier.
 */
function eruahaus_membership_requests_acf_form_validate_identifier( $valid, $value, $field, $input ) {
    // bail early if value is already invalid
    if ( !$valid ) {
        return $valid;
    }

    if ( $value !== $field['default_value'] ) {
        $valid = __( 'Incorrect form action', 'erua' );
    }

    return $valid;
}
add_filter('acf/validate_value/key=field_655aa1c002e35', 'eruahaus_membership_requests_acf_form_validate_identifier', 10, 4);
