<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

//================================================================================
// QUERYING
//================================================================================

/**
 * Add new query variables in the list of public query variables, 
 * so that we can use it in our custom URLs.
 * https://codex.wordpress.org/Function_Reference/get_query_var
 */
function add_public_query_vars_for_eruahaus_projects( $qvars ) {
    // Back-end ERUAHAUS Project listings
    $qvars[] = "eruahaus_project_meta_query";
    // Front-end ERUAHAUS Project listings
    $myvars = [
        "viewing",
        "keywords",
        "project_fa", # ERUAHAUS Project Category (Focus Area)
        "project_status",
        "project_universities",
        "project_fs",
        "project_sdgs"
    ];
    foreach ( $myvars as $key ) {
        if ( ! array_key_exists( $key, $qvars ) ) {
            $qvars[] = $key;
        }
    }

    return $qvars;
}
add_filter( 'query_vars', 'add_public_query_vars_for_eruahaus_projects' );

/**
 * Manipulate query before quering 
 * and map 'eruahaus_project_meta_query' custom query argument 
 * to the 'meta_query' WordPress's query argument.
 */
function eruahaus_project_pre_get_posts( $query ) {

    // if( function_exists('get_current_screen') && get_current_screen()->parent_base == 'edit' && is_admin() && $query->is_main_query() ) {
    //      //$query->query_vars['suppress_filters'] = true;
    //  }

    /**
     * If we're viewing the administration panel, 
     * and we're quering for "eruahaus-project" custom post type, 
     * and there's a "eruahaus_project_meta_query" custom query argument present, 
     * and its value is not empty.
     */
    if( is_admin() && get_query_var( 'post_type' ) == 'eruahaus-project' && ! empty( get_query_var( 'eruahaus_project_meta_query' ) ) && ! empty( $query->query_vars['eruahaus_project_meta_query'][0]['value'] ) && $query->is_main_query() ) {

        // Map 'eruahaus_project_meta_query' to 'meta_query'
        $query->set( 'meta_query', get_query_var( 'eruahaus_project_meta_query' ) );

        // echo '<pre>';
        // var_dump($query);
        // echo '</pre>';
        // die();

        // echo '<pre>';
        // print_r(get_query_var( 'eruahaus_project_meta_query' ));
        // echo '</pre>';
        // $temp = get_query_var( 'eruahaus_project_meta_query' );
        // echo $temp[0]["key"]; // assigned_to
        // die();

        // echo $query->query_vars['post_type'];
        // echo $query->query_vars['eruahaus_project_meta_query'][0]['key'];
        // echo '<pre>';
        // var_dump(get_query_var( 'eruahaus_project_meta_query[0][key]' ));
        // echo '</pre>';
        // die();

        //get_query_var( 'eruahaus_project_meta_query' )
        // array(1) {
        //     [0]=>
        //     array(1) {
        //       ["key"]=>
        //       string(11) "assigned_to"
        //     }
        //   }

        // if( !get_query_var( 'eruahaus_project_meta_query[0][value]' ) ) {
        //     echo "Yeah";
        //     // urldecode
        //     unset( $query->query_vars['eruahaus_project_meta_query[0][key]'] );
        //     remove_query_arg( 'eruahaus_project_meta_query', $query );
        //     remove_query_arg( urlencode('eruahaus_project_meta_query[0][key]'), $query );
        // }
    }

}
add_action( 'pre_get_posts', 'eruahaus_project_pre_get_posts' );

/**
 * Filter by custom taxonomy term and/or keyword, if present.
 * Choose from different query arguments depending on
 * the passed terms and keyword.
 * 
 * @return array $query_args An array of arguments to pass to WP_Query()
 */
function getERUAHAUSQueryArguments($project_focus_area_term = null, $project_status_terms = null, $project_university_terms = null, $project_funding_sources_terms = null, $project_sdg_terms = null, $keywords = null ) {

    // Get all projects by default
    $query_args = [
        'post_type' => ['eruahaus-project'],
        //'post_status' => ['publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit'],
        'post_status' => 'publish',
        'orderby'     => 'date',
        'numberposts' => -1,
        //'numberposts' => '5',
        'posts_per_page' => get_option('posts_per_page'),
        'paged' => ( get_query_var( 'paged' ) ) ? absint( get_query_var( 'paged' ) ) : 1,
    ];

    $tax_query = [];

    if ( !empty($project_focus_area_term) ) {
        $tax_query[] = [
            'taxonomy' => 'eruahaus-project-focus-area',
            'field' => 'term_id',
            'terms' => $project_focus_area_term,
            'operator' => 'IN',
        ];
    }

    if ( !empty($project_status_terms) ) {
        $tax_query[] = [
            'taxonomy' => 'eruahaus-project-status',
            'field' => 'term_id',
            'terms' => $project_status_terms,
            'operator' => 'IN',
        ];
    }

    if ( !empty($project_university_terms) ) {
        $tax_query[] = [
            'taxonomy' => 'eruahaus-erua-university',
            'field' => 'term_id',
            'terms' => $project_university_terms,
            'operator' => 'IN',
        ];
    }

    if ( !empty($project_funding_sources_terms) ) {
        $tax_query[] = [
            'taxonomy' => 'eruahaus-funding-source',
            'field' => 'term_id',
            'terms' => $project_funding_sources_terms,
            'operator' => 'IN',
        ];
    }

    if ( !empty($project_sdg_terms) ) {
        $tax_query[] = [
            'taxonomy' => 'eruahaus-sdg',
            'field' => 'term_id',
            'terms' => $project_sdg_terms,
            'operator' => 'IN',
        ];
    }

    /**
     * Important/Beware: ACF serialized meta data, meta querying with LIKE comparison = slow performance
     * Not all ACF meta is serialized in the database, so manually check the wp_postmeta table for the meta_value.
     * 
     * https://wordpress.stackexchange.com/questions/298156/meta-query-array-in-values-returns-an-error
     * https://wordpress.stackexchange.com/questions/16709/meta-query-with-meta-values-as-serialize-arrays
     * https://stackoverflow.com/questions/26364378/wp-query-when-meta-value-saved-as-serialized-array
     * https://support.advancedcustomfields.com/forums/topic/using-compare-in-operator-for-meta_query/
     */
    //$meta_query = [];

    // if ( !empty($project_funding_sources_terms) ) {
    //     $meta_query[] = [
    //         'key' => 'eruahaus_project_funding_source',
    //         'value' => $project_funding_sources_terms,
    //         'compare' => 'IN', # 'IN' for arrays or 'LIKE' for strings
    //     ];
    // }

    if ( !empty($keywords) ) {
        /**
         * Note: Prepending a keyword with a hyphen will exclude posts matching that keyword.
         * E.g., 'pillow -sofa' will return posts containing ‘pillow’ but not ‘sofa’.
         * 
         * https://developer.wordpress.org/reference/classes/wp_query/#search-parameters
         */
        $query_args['s'] = sanitize_text_field($keywords);
    }

    if ( !empty($tax_query) ) {
        if ( count($tax_query) > 1 ) $tax_query = ['relation' => 'AND'] + $tax_query;
        $query_args['tax_query'] = $tax_query;
    }

    // if ( !empty($meta_query) ) {
    //     if ( count($meta_query) > 1 ) $meta_query = ['relation' => 'OR'] + $meta_query;
    //     $query_args['meta_query'] = $meta_query;
    // }

    return $query_args;

}

//================================================================================
// LISTINGS SCREEN (edit.php) CUSTOMIZATION - COLUMNS, FILTERS etc.
//================================================================================

/**
 * Customize views @edit.php?post_type=eruahaus-project
 * More @ https://codex.wordpress.org/Plugin_API/Filter_Reference/views_edit-post
 */
function remove_views_from_eruahaus_projects($views) {  
    $user = wp_get_current_user();
    if ( in_array( 'administrator', (array) $user->roles ) ) {
        unset($views['mine']);
    }
    return $views;
}
add_filter('views_edit-eruahaus-project', 'remove_views_from_eruahaus_projects');

/**
 * Register columns for the back-end listings of the "eruahaus-project" custom post type.
 * https://codex.wordpress.org/Plugin_API/Filter_Reference/manage_$post_type_posts_columns
 */
function register_backend_eruahaus_project_columns($columns)
{
    $columns['status'] = __('Status', 'erua');
    $new_fields = [
        'applicant' => __('Applicant', 'erua'),
        'email' => __('Email', 'erua')
    ];
    associative_array_insert_after($columns, 'author', $new_fields);
    unset($columns['author']);
    // return [
    //     'cb' => '<input type="checkbox" />',
    //     'title' => __('Title', 'erua'),
    //     //'author' => __('Author', 'erua'),
    //     //'categories' => __('Categories', 'erua'),
    //     'applicant' => __('Applicant', 'erua'),
    //     'email' => __('Email', 'erua'),
    //     'topic' => __('Topic', 'erua'),
    //     'type' => __('Type', 'erua'),
    //     'stage' => __('Stage', 'erua'),
    //     'countries' => __('Countries', 'erua'),
    //     'languages' => __('Languages', 'erua'),
    //     'status' => __('Status', 'erua'),
    //     'assignee' => __('Assignee', 'erua'),
    //     'comments' => '<span class="vers comment-grey-bubble" title="' . esc_attr__('Comments', 'erua') . '"><span class="screen-reader-text">' . __('Comments', 'erua') . '</span></span>',
    //     'date' => __('Date', 'erua'),
    // ];
    return $columns;
}
add_filter('manage_eruahaus-project_posts_columns', 'register_backend_eruahaus_project_columns');

/**
 * Create the content of the custom columns
 * that were configured with register_backend_eruahaus_project_columns()
 * http://justintadlock.com/archives/2011/06/27/custom-columns-for-custom-post-types
 * https://wordpress.stackexchange.com/questions/253640/adding-custom-columns-to-custom-post-types
 * https://wpsmackdown.com/easy-filter-posts-wordpress-admin/
 * https://codex.wordpress.org/WordPress_Query_Vars
 * https://wordpress.stackexchange.com/questions/212519/filter-by-custom-field-in-custom-post-type-on-admin-page
 */
function manage_eruahaus_project_backend_custom_columns($column, $post_id)
{
    global $post;

    switch ($column) {

        case 'applicant':

            $author_id = $post->post_author;
            $author = get_the_author_meta('display_name', $author_id);

            if ($author_id) {
                printf('<a href="%s">%s</a>',
                        esc_url(add_query_arg(['post_type' => $post->post_type, 'author' => $author_id], 'edit.php')),
                        $author
                );
            }
            else {
                if ( get_field('field_6543bd41c96b3', $post_id) && get_field('field_6543bd4ac96b4', $post_id) ) {
                    $applicant_name = get_field('field_6543bd41c96b3', $post_id);
                    $applicant_email = get_field('field_6543bd4ac96b4', $post_id);

                    printf('<a href="%s">%s</a>',
                        esc_url(add_query_arg(['post_type' => $post->post_type, 'eruahaus_project_meta_query' => [ 
                            'relation' => 'AND', 
                            ['key' => 'eruahaus_project_applicant_name', 'value' => "$applicant_name", 'compare' => 'LIKE'], 
                            ['key' => 'eruahaus_project_applicant_email', 'value' => "$applicant_email", 'compare' => 'LIKE']
                        ]], 'edit.php')),
                        $applicant_name . ' (<span style="color: red;" title="' . __('Guest', 'erua') . '">G</span>)'
                    );
                }
                else {
                    echo __('Unknown', 'erua');
                }
            }

            break;

        case 'email':

            $author_id = $post->post_author;
            $author_email = get_the_author_meta('user_email', $author_id);

            if ($author_id) {
                printf('<a href="%s">%s</a>',
                        esc_url(add_query_arg(['post_type' => $post->post_type, 'author' => $author_id], 'edit.php')),
                        $author_email
                );
            }
            else {
                if ( get_field('field_6543bd41c96b3', $post_id) && get_field('field_6543bd4ac96b4', $post_id) ) {
                    $applicant_name = get_field('field_6543bd41c96b3', $post_id);
                    $applicant_email = get_field('field_6543bd4ac96b4', $post_id);

                    printf('<a href="%s">%s</a>',
                        esc_url(add_query_arg(['post_type' => $post->post_type, 'eruahaus_project_meta_query' => [ 
                            'relation' => 'AND', 
                            ['key' => 'eruahaus_project_applicant_name', 'value' => "$applicant_name", 'compare' => 'LIKE'], 
                            ['key' => 'eruahaus_project_applicant_email', 'value' => "$applicant_email", 'compare' => 'LIKE']
                        ]], 'edit.php')),
                        $applicant_email . ' (<span style="color: red;" title="' . __('Guest', 'erua') . '">G</span>)'
                    );
                }
                else {
                    echo __('Unknown', 'erua');
                }
            }

            break;

        /* If displaying the "status" column. */
        case 'status':

            /* Get the stages for the post. */
            $terms = get_the_terms($post_id, 'eruahaus-project-status');

            /* If no terms were found, output a default message. */
            if ( empty($terms) ) {
                echo __('Unknown', 'erua');
            }
            /* If there is a status */
            else {
                $out = array();
                foreach ($terms as $term) {
                    // "Active"
                    if ($term->term_id == 105) {
                        $class_name = 'project-status active-status';
                    }
                    // "Closed"
                    else if ($term->term_id == 106) {
                        $class_name = 'project-status closed-status';
                    }
                    else {
                        $class_name = 'project-status generic-status';
                    }

                    $out[] = sprintf('<a href="%s" class="%s">%s</a>',
                        esc_url(add_query_arg(['post_type' => $post->post_type, 'eruahaus-project-status' => $term->slug], 'edit.php')),
                        $class_name,
                        esc_html(sanitize_term_field('name', $term->name, $term->term_id, 'eruahaus-project-status', 'display'))
                    );
                }
                echo join(', ', $out);
            }

            break;

        /* Just break out of the switch statement for everything else. */
        default:
            break;
    }
}
add_action('manage_eruahaus-project_posts_custom_column', 'manage_eruahaus_project_backend_custom_columns', 10, 2);

/**
 * Add filters based on custom taxonomies,
 * for the project listings (@ edit.php).
 * https://generatewp.com/filtering-posts-by-taxonomies-in-the-dashboard/
 */
function create_backend_filters_for_projects($post_type, $which)
{

    // Apply this only on a specific post type
    if ('eruahaus-project' === $post_type) {

        /**
         * A list of taxonomy slugs to filter by
         * Note: Private and draft posts will be displayed 
         * but won't be counted in the taxonomy's terms.
         */
        $taxonomies = [
            'eruahaus-project-focus-area',
            'eruahaus-project-status',
            'eruahaus-erua-university',
            'eruahaus-funding-source',
            'eruahaus-sdg'
        ];

        foreach ($taxonomies as $taxonomy_slug) {

            // Retrieve taxonomy data
            $taxonomy_obj = get_taxonomy($taxonomy_slug);
            $taxonomy_name = $taxonomy_obj->labels->name;
            $taxonomy_is_hierarchical = $taxonomy_obj->hierarchical;

            if ( $taxonomy_is_hierarchical ) {

                /**
                 * Retrieve parent terms
                 */
                $top_level_terms = get_terms( [
                    'taxonomy'      => $taxonomy_slug,
                    'parent'        => '0',
                    'hide_empty'    => false,
                    'suppress_filters' => false
                ] );

                // Display filter HTML
                echo "<select name='{$taxonomy_slug}' id='{$taxonomy_slug}' class='postform'>";
                echo '<option value="">' . sprintf( esc_html__('All %s', 'erua'), $taxonomy_name ) . '</option>';
                foreach ( $top_level_terms as $top_level_term ) {

                    // View all projects with an awaiting nature (pending, waiting for x's reply etc.) of status
                    $query = get_posts(
                        [
                            'post_type' => $post_type,
                            'fields' => 'ids',
                            'post_status' => ['publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit'],
                            'numberposts' => -1,
                            'tax_query' => [
                                [
                                    'taxonomy' => $taxonomy_slug,
                                    'field' => 'slug',
                                    'terms' => $top_level_term->slug,
                                    'operator' => 'IN'
                                ]
                            ]
                        ]
                    );
                    wp_reset_postdata();
                    $count = count($query);

                    printf(
                        '<option value="%1$s" %2$s>%3$s (%4$s)</option>',
                        $top_level_term->slug,
                        ((isset($_GET[$taxonomy_slug]) && ($_GET[$taxonomy_slug] == $top_level_term->slug)) ? ' selected="selected"' : ''),
                        $top_level_term->name,
                        $count //$top_level_term->count
                    );

                    $top_term_id = $top_level_term->term_id;

                    $second_level_terms = get_terms( array(
                        'taxonomy' => $taxonomy_slug, # you could also use $taxonomy as defined in the first lines
                        'child_of' => $top_term_id,
                        'parent' => $top_term_id, # disable this line to see more child elements (child-child-child-terms)
                        'hide_empty' => false,
                    ) );
                    foreach ($second_level_terms as $second_level_term) {

                        $query = get_posts(
                            [
                                'post_type' => $post_type,
                                'fields' => 'ids',
                                'post_status' => ['publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit'],
                                'numberposts' => -1,
                                'tax_query' => [
                                    [
                                        'taxonomy' => $taxonomy_slug,
                                        'field' => 'slug',
                                        'terms' => $second_level_term->slug,
                                        'operator' => 'IN'
                                    ]
                                ]
                            ]
                        );
                        wp_reset_postdata();
                        $count = count($query);

                        printf(
                            '<option value="%1$s" %2$s>%3$s (%4$s)</option>',
                            $second_level_term->slug,
                            ((isset($_GET[$taxonomy_slug]) && ($_GET[$taxonomy_slug] == $second_level_term->slug)) ? ' selected="selected"' : ''),
                            '&nbsp;&nbsp;&nbsp;&nbsp;' . $second_level_term->name,
                            $count//$second_level_term->count
                        );
                    }
                }
                echo '</select>';
            }
            else {

                /**
                 * Retrieve parent terms
                 */
                $top_level_terms = get_terms( [
                    'taxonomy'      => $taxonomy_slug,
                    'parent'        => '0',
                    'hide_empty'    => false,
                    'suppress_filters' => false
                ] );

                // Display filter HTML
                echo "<select name='{$taxonomy_slug}' id='{$taxonomy_slug}' class='postform'>";
                echo '<option value="">' . sprintf( esc_html__('Show All %s', 'erua'), $taxonomy_name ) . '</option>';
                foreach ( $top_level_terms as $top_level_term ) {

                    // View all projects with an awaiting nature (pending, waiting for x's reply etc.) of status
                    $query = get_posts(
                        [
                            'post_type' => $post_type,
                            'fields' => 'ids',
                            'post_status' => ['publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit'],
                            'numberposts' => -1,
                            'tax_query' => [
                                [
                                    'taxonomy' => $taxonomy_slug,
                                    'field' => 'slug',
                                    'terms' => $top_level_term->slug,
                                    'operator' => 'IN'
                                ]
                            ]
                        ]
                    );
                    wp_reset_postdata();
                    $count = count($query);

                    printf(
                        '<option value="%1$s" %2$s>%3$s (%4$s)</option>',
                        $top_level_term->slug,
                        ((isset($_GET[$taxonomy_slug]) && ($_GET[$taxonomy_slug] == $top_level_term->slug)) ? ' selected="selected"' : ''),
                        $top_level_term->name,
                        $count //$top_level_term->count
                    );
                }
                echo '</select>';
            }
        }

    }

}
add_action('restrict_manage_posts', 'create_backend_filters_for_projects', 10, 2);

//================================================================================
// BACK-END FIELD HANDLING
//================================================================================

/**
 * If a post of "eruahaus-project" type was updated,
 * then adjust the "date_updated" custom field.
 */
function eruahaus_project_post_update_modified_date($post_id, $post, $update)
{
    if ( $post->post_type == 'eruahaus-project' ) {

        if ( $update ) {
            // If the "date_updated" custom field exists, update it.
            if ( get_field('field_6543bd3bc96b2') ) {
                update_field('field_6543bd3bc96b2', get_the_modified_time('j F Y, H:i', $post_id), $post_id);
            }
        }
        else if ( !$update ) {
            // If the "date_created" custom field value does not exist, initialiaze it.
            if ( !get_field('field_6543bd37c96b1') ) {
                update_field( 'field_6543bd37c96b1', get_post_time('j F Y, H:i', false, $post_id, false ), $post_id );
            }
            // If the "date_updated" custom field value does not exist, initialiaze it.
            if ( !get_field('field_6543bd3bc96b2') ) {
                update_field( 'field_6543bd3bc96b2', get_post_time('j F Y, H:i', false, $post_id, false ), $post_id );
            }
        }

        // Update eruahaus-sdg taxonomy terms based on the SDG field data
        if ( isset($_POST['acf']['field_654a5549c7a11']) ) {
            $sdg_terms = $_POST['acf']['field_654a5549c7a11'];
            $sdg_terms_ids = array_map('intval', $sdg_terms);
            wp_set_object_terms( (int) $post_id, $sdg_terms_ids, 'eruahaus-sdg', false );
            //wp_set_post_terms( (int) $post_id, $sdg_terms_ids, 'eruahaus-sdg', false );
        }

    }

}
add_action('save_post', 'eruahaus_project_post_update_modified_date', 10, 3);

/**
 * Load choices for SDGs.
 *
 * https://www.advancedcustomfields.com/resources/acf-load_value/
 */
function eruahaus_project_load_sdg_choices( $value, $post_id, $field ) {

    $sdg_terms = get_the_terms($post_id, 'eruahaus-sdg');

    if ( $sdg_terms ) {
        // Reset values
        $value = [];

        foreach ( $sdg_terms as $key => $sdg ) {
            $value[] = strval($sdg->term_id);
        }

    }

    return $value;
}
add_filter('acf/load_value/key=field_654a5549c7a11', 'eruahaus_project_load_sdg_choices', 10, 3); # SDGs

function eruahaus_project_load_applicant_name( $value, $post_id, $field ) {

    $user_id = get_field('eruahaus_project_registered_user_id', $post_id);

    if ( !$user_id ) {
        return $value;
    }

    $user = get_user_by('id', (int) $user_id);

    if ( $value != $user->display_name ) {
        $value = $user->display_name;
    }

    return $value;
}
add_filter('acf/load_value/key=field_6543bd41c96b3', 'eruahaus_project_load_applicant_name', 10, 3); # Applicant's Name

function eruahaus_project_load_applicant_email( $value, $post_id, $field ) {

    $user_id = get_field('eruahaus_project_registered_user_id', $post_id);

    if ( !$user_id ) {
        return $value;
    }

    $user = get_user_by('id', (int) $user_id);

    if ( $value != $user->user_email ) {
        $value = $user->user_email;
    }

    return $value;
}
add_filter('acf/load_value/key=field_6543bd4ac96b4', 'eruahaus_project_load_applicant_email', 10, 3); # Applicant's E-mail

function eruahaus_project_load_applicant_profile_url( $value, $post_id, $field ) {

    if ( function_exists('bp_is_active') ) {

        $user_id = get_field('eruahaus_project_registered_user_id', $post_id);

        if ( !$user_id ) {
            return $value;
        }

        $user_profile_url = bbp_get_user_profile_url( (int) $user_id );

        if ( $value != $user_profile_url ) {
            $value = $user_profile_url;
        }
    }

    return $value;
}
add_filter('acf/load_value/key=field_655b8e5149d2c', 'eruahaus_project_load_applicant_profile_url', 10, 3); # Applicant's Profile URL

function eruahaus_project_load_project_group_url( $value, $post_id, $field ) {

    if ( function_exists('bp_is_active') ) {

        $group_id = get_field('eruahaus_project_group_id', $post_id);

        if ( !$group_id ) {
            return $value;
        }

        // https://stackoverflow.com/a/65874506
        $group_obj = groups_get_group( (int) $group_id );
        $group_url = bp_get_group_permalink( $group_obj );

        if ( $value != $group_url ) {
            $value = $group_url;
        }
    }

    return $value;
}
add_filter('acf/load_value/key=field_655c374abc657', 'eruahaus_project_load_project_group_url', 10, 3); # Project's Group URL

function eruahaus_project_render_applicant_profile_url( $field ) {
    // echo '<pre>';
    // var_dump($field);
    // echo '</pre>';
    if ( $field['value'] ) {
        echo '<a href="'.$field['value'].'" target="_blank" style="display: inline-block; background-color: aliceblue; padding: 5px; border-radius: 0 0 5px 5px; text-decoration: none;">'.__('Visit profile', 'erua').'</a>';
    }
}
add_action( 'acf/render_field/key=field_655b8e5149d2c', 'eruahaus_project_render_applicant_profile_url', 10, 1 );

function eruahaus_project_render_project_group_url( $field ) {
    // echo '<pre>';
    // var_dump($field);
    // echo '</pre>';
    if ( $field['value'] ) {
        echo '<a href="'.$field['value'].'" target="_blank" style="display: inline-block; background-color: aliceblue; padding: 5px; border-radius: 0 0 5px 5px; text-decoration: none;">'.__('Visit Team Space', 'erua').'</a>';
    }
}
add_action( 'acf/render_field/key=field_655c374abc657', 'eruahaus_project_render_project_group_url', 10, 1 );

function eruahaus_project_acf_hide_valueless_field( $field ) {

    // Don't show this field if it doesn't contain a value.
    if( !$field['value'] ) {
        return false;
    }
    return $field;
}
add_filter('acf/prepare_field/key=field_655bfefa672b0', 'eruahaus_project_acf_hide_valueless_field'); # Team Space Name
add_filter('acf/prepare_field/key=field_655bfeff672b1', 'eruahaus_project_acf_hide_valueless_field'); # Team Space Description
add_filter('acf/prepare_field/key=field_655bfaa72dc4a', 'eruahaus_project_acf_hide_valueless_field'); # Team Space ID
add_filter('acf/prepare_field/key=field_655c374abc657', 'eruahaus_project_acf_hide_valueless_field'); # Team Space URL

//================================================================================
// DISABLING STUFF
//================================================================================

/**
 * Make an ACF field readonly to prevent editing.
 * https://www.advancedcustomfields.com/resources/acf-load_field/
 * https://support.advancedcustomfields.com/forums/topic/read-only-field-2/
 */
function eruahaus_project_acf_make_field_readonly( $field ) {

    // $field['choices'] = array(
    //     'custom' => 'My Custom Choice'
    // );
    //$field['disabled'] = 1;
    $field['readonly'] = true;

    return $field;

}
/**
 * Make an ACF field disabled to prevent editing.
 * https://www.advancedcustomfields.com/resources/acf-load_field/
 * https://github.com/AdvancedCustomFields/acf/issues/26
 */
function eruahaus_project_acf_make_field_disabled( $field ) {

    $field['disabled'] = true;

    return $field;

}
add_filter('acf/load_field/key=field_6543bd41c96b3', 'eruahaus_project_acf_make_field_readonly'); # Applicant's Name
add_filter('acf/load_field/key=field_6543bd4fc96b5', 'eruahaus_project_acf_make_field_readonly'); # Applicant's User ID
add_filter('acf/load_field/key=field_6543bd37c96b1', 'eruahaus_project_acf_make_field_readonly'); # Date created
add_filter('acf/load_field/key=field_6543bd3bc96b2', 'eruahaus_project_acf_make_field_readonly'); # Date updated
add_filter('acf/load_field/key=field_6543bd56c96b7', 'eruahaus_project_acf_make_field_readonly'); # Unique Token
add_filter('acf/load_field/key=field_655bfefa672b0', 'eruahaus_project_acf_make_field_readonly'); # Team Space Name
add_filter('acf/load_field/key=field_655bfeff672b1', 'eruahaus_project_acf_make_field_readonly'); # Team Space Description
add_filter('acf/load_field/key=field_655bfaa72dc4a', 'eruahaus_project_acf_make_field_readonly'); # Team Space ID
if ( !current_user_can('administrator') ) {
    add_filter('acf/load_field/key=field_6543bd4ac96b4', 'eruahaus_project_acf_make_field_readonly'); # Applicant's E-mail
}

/**
 * Disable various fields at a Project's editing screen.
 */
function erua_disable_eruahaus_project_fields() {
    global $pagenow;

    // If we are at the back-office, at a Project post.
    if ( is_admin() && !( defined( 'DOING_AJAX' ) && DOING_AJAX ) && ($pagenow == 'post.php' || $pagenow == 'post-new.php') && get_post_type() == 'eruahaus-project' ) {
    ?>
        <script>
        jQuery(document).ready(function() {
            // Disable ACF's date picker at Project's editing screen.
            jQuery('#eruahaus-project-date-created .hasDatepicker').prop('disabled', 'disabled');
            jQuery('#eruahaus-project-date-updated .hasDatepicker').prop('disabled', 'disabled');

            jQuery("div[data-key='field_6543bd37c96b1']").css({position: "relative"});
            jQuery("div[data-key='field_6543bd37c96b1']").prepend('<div id="date-created-tracking-jammer" style="display:block; position:absolute; top:0; left:0; width:100%; height:100%; cursor: initial; z-index:9; background-color: aliceblue;"></div>');
            jQuery("div[data-key='field_6543bd3bc96b2']").css({position: "relative"});
            jQuery("div[data-key='field_6543bd3bc96b2']").prepend('<div id="date-updated-tracking-jammer" style="display:block; position:absolute; top:0; left:0; width:100%; height:100%; cursor: initial; z-index:9; background-color: aliceblue;"></div>');

            // Disable various Publishing actions
            jQuery('body.wp-admin #misc-publishing-actions .edit-timestamp').css({display: "none"});

            <?php //if ( is_user_logged_in() && !current_user_can('administrator') ) : ?>
            // Disable Title
            //jQuery('#title').prop('disabled', true);
            // Disable various Publishing actions
            jQuery('body.wp-admin #misc-publishing-actions #major-publishing-actions').css({display: "none"});

            // Hide Status Tracking
            jQuery("div[data-key='field_6543bd53c96b6']").css({position: "relative"});
            jQuery("div[data-key='field_6543bd53c96b6']").prepend('<div id="status-tracking-jammer" style="display:block; position:absolute; top:0; left:0; width:100%; height:100%; cursor: initial; z-index:9; background-color: aliceblue;"></div>');

            // Hide Unique ID
            jQuery("div[data-key='field_6543bd56c96b7']").css({position: "relative"});
            jQuery("div[data-key='field_6543bd56c96b7']").prepend('<div id="unique-id-jammer" style="display:block; position:absolute; top:0; left:0; width:100%; height:100%; cursor: initial; z-index:9; background-color: aliceblue;"></div>');

            <?php //endif; ?>
        });
        </script>
    <?php
    }

}
add_action('admin_print_footer_scripts', 'erua_disable_eruahaus_project_fields');

//================================================================================
// BUBBLE NOTIFICATIONS
//================================================================================

/**
 * Notification "bubble" with pending/open Projects count
 */
add_action('admin_menu', function () {
    global $menu;
    //$count_posts = wp_count_posts('eruahaus-project');
    //$count = $count_posts->pending;

    // View all projects with an awaiting nature (pending, waiting for x's reply etc.) of status
    $projects_query = get_posts(
        [
            'post_type' => 'eruahaus-project',
            'fields' => 'ids',
            //'post_status' => 'any', // Published, Pending etc.
            //'post_status' => ['publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit', 'trash'],
            //'post_status' => ['publish', 'pending', 'draft', 'future', 'private', 'inherit'],
            'post_status' => ['pending', 'draft'],
            'numberposts' => -1,
            // All projects that are not active or closed i.e. pending or open
            // 'tax_query' => [
            //     [
            //         'taxonomy' => 'eruahaus-project-status',
            //         'field' => 'slug',
            //         'terms' => ['closed', 'active'],
            //         'operator' => 'NOT IN',
            //     ],
            // ],
        ]
    );
    if (!empty($projects_query)) wp_reset_postdata();
    $count = count($projects_query);
    /* echo '<pre>';
    var_dump($projects_query);
    echo '</pre>';
    die(); */

    $menu_item = wp_list_filter(
        $menu,
        [2 => 'edit.php?post_type=eruahaus-project'] // 2 is the position of an array item which contains URL, it will always be 2!
    );
    if ( !empty($menu_item) && $count > 0 ) {
        $menu_item_position = key($menu_item); // get the array key (position) of the element
        $menu[$menu_item_position][0] .= ' <span class="awaiting-mod" title="' . $count . ' Projects require actions">' . $count . '</span>';
    }
});
